<?php
header('Content-Type: application/json');
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/SupabaseClient.php';
require_once '../includes/TableManager.php';

try {
    // Validate request method
    if (getRequestMethod() !== 'POST') {
        throw new Exception('Only POST method allowed');
    }
    
    // Get JSON input
    $input = getJsonInput();
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validate required parameters
    validateRequired($input, ['table', 'data']);
    
    $table = $input['table'];
    $data = $input['data'];
    
    if (!is_array($data) || empty($data)) {
        throw new Exception('Data must be a non-empty array');
    }
    
    // Initialize Supabase client
    $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
    $tableManager = new TableManager($supabase);
    
    // Sanitize table name
    $table = $tableManager->sanitizeTableName($table);
    
    // Validate table exists
    $columns = $tableManager->getTableColumns($table);
    
    if (empty($columns)) {
        throw new Exception('Table not found');
    }
    
    // Validate and sanitize data
    $sanitizedData = [];
    foreach ($data as $column => $value) {
        $sanitizedColumn = $tableManager->sanitizeColumnName($column);
        
        if (in_array($sanitizedColumn, $columns)) {
            $sanitizedData[$sanitizedColumn] = $value;
        }
    }
    
    if (empty($sanitizedData)) {
        throw new Exception('No valid data provided');
    }
    
    // Add the row
    $result = $tableManager->addRow($table, $sanitizedData);
    
    logError("Row added successfully", [
        'table' => $table,
        'columns' => array_keys($sanitizedData)
    ]);
    
    successResponse([
        'added' => true,
        'table' => $table,
        'data' => $result
    ], 'Row added successfully');
    
} catch (Exception $e) {
    logError("Error adding row: " . $e->getMessage(), [
        'input' => $input ?? null
    ]);
    errorResponse($e->getMessage());
}
?>
