<?php
header('Content-Type: application/json');
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/SupabaseClient.php';
require_once '../includes/TableManager.php';

try {
    // Validate request method
    if (getRequestMethod() !== 'POST') {
        throw new Exception('Only POST method allowed');
    }
    
    // Get JSON input
    $input = getJsonInput();
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validate required parameters
    validateRequired($input, ['table', 'rows']);
    
    $table = $input['table'];
    $rowIds = $input['rows'];
    
    if (!is_array($rowIds) || empty($rowIds)) {
        throw new Exception('Row IDs must be a non-empty array');
    }
    
    // Initialize Supabase client
    $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
    $tableManager = new TableManager($supabase);
    
    // Sanitize table name
    $table = $tableManager->sanitizeTableName($table);
    
    // Get primary key for the table
    $primaryKey = $tableManager->getPrimaryKey($table);
    
    // Delete multiple rows
    $deletedCount = 0;
    $errors = [];
    
    foreach ($rowIds as $rowId) {
        try {
            $result = $tableManager->deleteRow($table, $rowId, $primaryKey);
            if ($result) {
                $deletedCount++;
            }
        } catch (Exception $e) {
            $errors[] = "Row {$rowId}: " . $e->getMessage();
        }
    }
    
    logError("Bulk delete completed", [
        'table' => $table,
        'requested' => count($rowIds),
        'deleted' => $deletedCount,
        'errors' => count($errors)
    ]);
    
    successResponse([
        'deleted_count' => $deletedCount,
        'total_requested' => count($rowIds),
        'errors' => $errors,
        'success' => $deletedCount > 0
    ]);
    
} catch (Exception $e) {
    logError("Error bulk deleting rows: " . $e->getMessage());
    errorResponse($e->getMessage());
}
?>
