<?php
header('Content-Type: application/json');
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/SupabaseClient.php';
require_once '../includes/TableManager.php';

try {
    // Validate request method
    if (getRequestMethod() !== 'POST') {
        throw new Exception('Only POST method allowed');
    }
    
    // Get JSON input
    $input = getJsonInput();
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validate required parameters
    validateRequired($input, ['table', 'row_id', 'column', 'value']);
    
    $table = $input['table'];
    $rowId = $input['row_id'];
    $column = $input['column'];
    $value = $input['value'];
    $primaryKey = arrayGet($input, 'primary_key', 'id');
    
    // Initialize Supabase client
    $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
    $tableManager = new TableManager($supabase);
    
    // Sanitize inputs
    $table = $tableManager->sanitizeTableName($table);
    $column = $tableManager->sanitizeColumnName($column);
    $primaryKey = $tableManager->sanitizeColumnName($primaryKey);
    
    // Validate table and column exist
    $columns = $tableManager->getTableColumns($table);
    
    if (empty($columns)) {
        throw new Exception('Table not found');
    }
    
    if (!in_array($column, $columns)) {
        throw new Exception('Column not found in table');
    }
    
    // Update the cell
    $result = $tableManager->updateCell($table, $rowId, $column, $value, $primaryKey);
    
    logError("Cell updated successfully", [
        'table' => $table,
        'row_id' => $rowId,
        'column' => $column,
        'old_value' => 'hidden',
        'new_value' => 'hidden'
    ]);
    
    successResponse([
        'updated' => true,
        'table' => $table,
        'row_id' => $rowId,
        'column' => $column,
        'new_value' => $value
    ], 'Cell updated successfully');
    
} catch (Exception $e) {
    logError("Error updating cell: " . $e->getMessage(), [
        'input' => $input ?? null
    ]);
    errorResponse($e->getMessage());
}
?>
