<?php

class SupabaseClient {
    private $url;
    private $anonKey;
    private $serviceKey;
    
    public function __construct($url, $anonKey, $serviceKey) {
        $this->url = rtrim($url, '/');
        $this->anonKey = $anonKey;
        $this->serviceKey = $serviceKey;
    }
    
    /**
     * Make a request to Supabase REST API
     */
    private function makeRequest($endpoint, $method = 'GET', $data = null, $useServiceKey = false) {
        $url = $this->url . '/rest/v1/' . ltrim($endpoint, '/');
        
        $headers = [
            'Content-Type: application/json',
            'Prefer: return=representation',
            'apikey: ' . ($useServiceKey ? $this->serviceKey : $this->anonKey),
            'Authorization: Bearer ' . ($useServiceKey ? $this->serviceKey : $this->anonKey)
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ]);
        
        if ($data && in_array($method, ['POST', 'PATCH', 'PUT'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }
        
        $decodedResponse = json_decode($response, true);
        
        if ($httpCode >= 400) {
            $errorMessage = isset($decodedResponse['message']) ? 
                $decodedResponse['message'] : 
                "HTTP Error: " . $httpCode;
            throw new Exception($errorMessage);
        }
        
        return $decodedResponse;
    }
    
    /**
     * Get all available tables
     */
    public function getTables() {
        try {
            $response = $this->makeRequest('', 'GET', null, true);
            
            // Extract table names from the response
            $tables = [];
            if (isset($response['paths'])) {
                foreach ($response['paths'] as $path => $info) {
                    if (strpos($path, '/') === 0 && substr_count($path, '/') === 1) {
                        $tableName = ltrim($path, '/');
                        if ($tableName && !in_array($tableName, ['rpc'])) {
                            $tables[] = $tableName;
                        }
                    }
                }
            }
            
            // Fallback: try to get schema information
            if (empty($tables)) {
                $schemaResponse = $this->makeRequest('?select=table_name&table_schema=eq.public', 'GET', null, true);
                if (is_array($schemaResponse)) {
                    foreach ($schemaResponse as $table) {
                        if (isset($table['table_name'])) {
                            $tables[] = $table['table_name'];
                        }
                    }
                }
            }
            
            return array_unique($tables);
        } catch (Exception $e) {
            error_log("Error getting tables: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get table data with optional filters
     */
    public function getTableData($tableName, $filters = [], $limit = 50, $offset = 0) {
        $query = "?select=*";
        
        // Add filters
        foreach ($filters as $column => $value) {
            if (!empty($value)) {
                $query .= "&" . urlencode($column) . "=ilike.*" . urlencode($value) . "*";
            }
        }
        
        // Add pagination
        $query .= "&limit=" . intval($limit);
        $query .= "&offset=" . intval($offset);
        
        return $this->makeRequest($tableName . $query);
    }
    
    /**
     * Get table structure/columns
     */
    public function getTableStructure($tableName) {
        try {
            // Get a single row to determine column structure
            $sampleData = $this->makeRequest($tableName . "?limit=1");
            
            if (!empty($sampleData) && is_array($sampleData)) {
                return array_keys($sampleData[0]);
            }
            
            return [];
        } catch (Exception $e) {
            error_log("Error getting table structure: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Update a single cell value
     */
    public function updateCell($tableName, $rowId, $column, $value, $primaryKey = 'id') {
        $data = [$column => $value];
        $endpoint = $tableName . "?" . urlencode($primaryKey) . "=eq." . urlencode($rowId);
        
        return $this->makeRequest($endpoint, 'PATCH', $data, true);
    }
    
    /**
     * Delete a row
     */
    public function deleteRow($tableName, $rowId, $primaryKey = 'id') {
        $endpoint = $tableName . "?" . urlencode($primaryKey) . "=eq." . urlencode($rowId);
        
        return $this->makeRequest($endpoint, 'DELETE', null, true);
    }
    
    /**
     * Insert a new row
     */
    public function insertRow($tableName, $data) {
        return $this->makeRequest($tableName, 'POST', $data, true);
    }
    
    /**
     * Get row count for pagination
     */
    public function getRowCount($tableName, $filters = []) {
        $query = "?select=count";
        
        // Add filters
        foreach ($filters as $column => $value) {
            if (!empty($value)) {
                $query .= "&" . urlencode($column) . "=ilike.*" . urlencode($value) . "*";
            }
        }
        
        try {
            $response = $this->makeRequest($tableName . $query . "&head=true");
            // The count is usually returned in headers, but we'll do a simple count
            $allData = $this->getTableData($tableName, $filters, 99999, 0);
            return count($allData);
        } catch (Exception $e) {
            return 0;
        }
    }
}
?>
