<?php

require_once 'SupabaseClient.php';

class TableManager {
    private $supabase;
    
    public function __construct($supabaseClient) {
        $this->supabase = $supabaseClient;
    }
    
    /**
     * Get all available tables
     */
    public function getAllTables() {
        return $this->supabase->getTables();
    }
    
    /**
     * Get table data with filters and pagination
     */
    public function getTableData($tableName, $filters = [], $page = 1, $limit = DEFAULT_LIMIT) {
        $offset = ($page - 1) * $limit;
        return $this->supabase->getTableData($tableName, $filters, $limit, $offset);
    }
    
    /**
     * Get table columns
     */
    public function getTableColumns($tableName) {
        return $this->supabase->getTableStructure($tableName);
    }
    
    /**
     * Update a cell value
     */
    public function updateCell($tableName, $rowId, $column, $value, $primaryKey = 'id') {
        try {
            // Validate input
            if (empty($tableName) || empty($column)) {
                throw new Exception("Table name and column are required");
            }
            
            // Convert value to appropriate type
            $value = $this->convertValue($value);
            
            return $this->supabase->updateCell($tableName, $rowId, $column, $value, $primaryKey);
        } catch (Exception $e) {
            error_log("Error updating cell: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Delete a row
     */
    public function deleteRow($tableName, $rowId, $primaryKey = 'id') {
        try {
            if (empty($tableName) || empty($rowId)) {
                throw new Exception("Table name and row ID are required");
            }
            
            return $this->supabase->deleteRow($tableName, $rowId, $primaryKey);
        } catch (Exception $e) {
            error_log("Error deleting row: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Add a new row
     */
    public function addRow($tableName, $data) {
        try {
            if (empty($tableName) || empty($data)) {
                throw new Exception("Table name and data are required");
            }
            
            // Process and validate data
            $processedData = [];
            foreach ($data as $column => $value) {
                if (!empty($value) || $value === '0' || $value === 0) {
                    $processedData[$column] = $this->convertValue($value);
                }
            }
            
            return $this->supabase->insertRow($tableName, $processedData);
        } catch (Exception $e) {
            error_log("Error adding row: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Get row count for pagination
     */
    public function getRowCount($tableName, $filters = []) {
        return $this->supabase->getRowCount($tableName, $filters);
    }
    
    /**
     * Convert string values to appropriate types
     */
    private function convertValue($value) {
        if ($value === null || $value === '') {
            return null;
        }
        
        // Handle boolean values
        if (strtolower($value) === 'true') {
            return true;
        }
        if (strtolower($value) === 'false') {
            return false;
        }
        
        // Handle numeric values
        if (is_numeric($value)) {
            if (strpos($value, '.') !== false) {
                return floatval($value);
            }
            return intval($value);
        }
        
        // Handle JSON strings
        if ($this->isJson($value)) {
            return json_decode($value, true);
        }
        
        return $value;
    }
    
    /**
     * Check if a string is valid JSON
     */
    private function isJson($string) {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
    
    /**
     * Sanitize table name to prevent SQL injection
     */
    public function sanitizeTableName($tableName) {
        return preg_replace('/[^a-zA-Z0-9_]/', '', $tableName);
    }
    
    /**
     * Sanitize column name to prevent SQL injection
     */
    public function sanitizeColumnName($columnName) {
        return preg_replace('/[^a-zA-Z0-9_]/', '', $columnName);
    }
    
    /**
     * Get primary key column for a table
     */
    public function getPrimaryKey($tableName) {
        // Default to 'id', but in a real implementation you might want to
        // query the database schema to get the actual primary key
        return 'id';
    }
    
    /**
     * Format value for display
     */
    public function formatValueForDisplay($value) {
        if ($value === null) {
            return '<em>NULL</em>';
        }
        
        if (is_bool($value)) {
            return $value ? 'true' : 'false';
        }
        
        if (is_array($value) || is_object($value)) {
            return json_encode($value, JSON_PRETTY_PRINT);
        }
        
        return htmlspecialchars($value);
    }
    
    /**
     * Validate filters
     */
    public function validateFilters($filters, $validColumns) {
        $validatedFilters = [];
        
        foreach ($filters as $column => $value) {
            $sanitizedColumn = $this->sanitizeColumnName($column);
            if (in_array($sanitizedColumn, $validColumns) && !empty($value)) {
                $validatedFilters[$sanitizedColumn] = $value;
            }
        }
        
        return $validatedFilters;
    }
    
    /**
     * Search across all columns in a table
     */
    public function searchTable($tableName, $query, $columns) {
        try {
            // Build search filters for each column
            $searchResults = [];
            
            // Search each column for the query term
            foreach ($columns as $column) {
                $filters = [$column => $query];
                $results = $this->supabase->getTableData($tableName, $filters, 100, 0);
                
                if (!empty($results)) {
                    $searchResults = array_merge($searchResults, $results);
                }
            }
            
            // Remove duplicates based on primary key
            $primaryKey = $this->getPrimaryKey($tableName);
            $uniqueResults = [];
            $seenIds = [];
            
            foreach ($searchResults as $row) {
                $id = $row[$primaryKey] ?? null;
                if ($id && !in_array($id, $seenIds)) {
                    $uniqueResults[] = $row;
                    $seenIds[] = $id;
                }
            }
            
            return $uniqueResults;
            
        } catch (Exception $e) {
            error_log("Error searching table: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Export table data to CSV or JSON
     */
    public function exportTableData($tableName, $data, $format) {
        try {
            $timestamp = date('Y-m-d_H-i-s');
            $filename = $tableName . '_export_' . $timestamp . '.' . $format;
            $filepath = __DIR__ . '/../temp/exports/' . $filename;
            
            // Ensure directory exists
            $dir = dirname($filepath);
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
            
            if ($format === 'csv') {
                $this->exportToCSV($data, $filepath);
            } elseif ($format === 'json') {
                $this->exportToJSON($data, $filepath);
            } else {
                throw new Exception('Unsupported export format');
            }
            
            return $filename;
            
        } catch (Exception $e) {
            error_log("Error exporting table data: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Export data to CSV format
     */
    private function exportToCSV($data, $filepath) {
        $file = fopen($filepath, 'w');
        
        if (empty($data)) {
            fclose($file);
            return;
        }
        
        // Write headers
        $headers = array_keys($data[0]);
        fputcsv($file, $headers);
        
        // Write data rows
        foreach ($data as $row) {
            fputcsv($file, $row);
        }
        
        fclose($file);
    }
    
    /**
     * Export data to JSON format
     */
    private function exportToJSON($data, $filepath) {
        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        file_put_contents($filepath, $json);
    }
    
    /**
     * Get table statistics
     */
    public function getTableStatistics($tableName) {
        try {
            $stats = [];
            $columns = $this->getTableColumns($tableName);
            
            // Get total row count
            $stats['total_rows'] = $this->getRowCount($tableName);
            $stats['total_columns'] = count($columns);
            $stats['columns'] = $columns;
            
            // Get sample data to analyze column types
            $sampleData = $this->supabase->getTableData($tableName, [], 10, 0);
            $stats['column_types'] = [];
            
            if (!empty($sampleData)) {
                foreach ($columns as $column) {
                    $types = [];
                    foreach ($sampleData as $row) {
                        $value = $row[$column] ?? null;
                        $types[] = gettype($value);
                    }
                    $stats['column_types'][$column] = array_count_values($types);
                }
            }
            
            return $stats;
            
        } catch (Exception $e) {
            error_log("Error getting table statistics: " . $e->getMessage());
            throw $e;
        }
    }
}
?>
