<?php

/**
 * Utility functions for the Supabase Table Editor
 */

/**
 * Sanitize HTML output
 */
function h($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Generate CSRF token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Return JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Return error response
 */
function errorResponse($message, $statusCode = 400) {
    jsonResponse(['error' => $message], $statusCode);
}

/**
 * Return success response
 */
function successResponse($data = null, $message = 'Success') {
    $response = ['success' => true, 'message' => $message];
    if ($data !== null) {
        $response['data'] = $data;
    }
    jsonResponse($response);
}

/**
 * Validate required parameters
 */
function validateRequired($params, $required) {
    $missing = [];
    foreach ($required as $param) {
        if (!isset($params[$param]) || $params[$param] === '') {
            $missing[] = $param;
        }
    }
    
    if (!empty($missing)) {
        throw new Exception("Missing required parameters: " . implode(', ', $missing));
    }
}

/**
 * Get request method
 */
function getRequestMethod() {
    return $_SERVER['REQUEST_METHOD'] ?? 'GET';
}

/**
 * Get request body as JSON
 */
function getJsonInput() {
    $input = file_get_contents('php://input');
    return json_decode($input, true);
}

/**
 * Log error message
 */
function logError($message, $context = []) {
    $logMessage = date('Y-m-d H:i:s') . " - " . $message;
    if (!empty($context)) {
        $logMessage .= " - Context: " . json_encode($context);
    }
    error_log($logMessage);
}

/**
 * Format file size
 */
function formatBytes($size, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    
    return round($size, $precision) . ' ' . $units[$i];
}

/**
 * Check if request is AJAX
 */
function isAjaxRequest() {
    return isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Generate pagination info
 */
function generatePagination($currentPage, $totalRows, $limit) {
    $totalPages = ceil($totalRows / $limit);
    
    return [
        'current_page' => $currentPage,
        'total_pages' => $totalPages,
        'total_rows' => $totalRows,
        'limit' => $limit,
        'has_next' => $currentPage < $totalPages,
        'has_prev' => $currentPage > 1,
        'next_page' => $currentPage + 1,
        'prev_page' => $currentPage - 1
    ];
}

/**
 * Truncate text for display
 */
function truncateText($text, $length = 50, $suffix = '...') {
    if (strlen($text) <= $length) {
        return $text;
    }
    
    return substr($text, 0, $length - strlen($suffix)) . $suffix;
}

/**
 * Detect data type
 */
function detectDataType($value) {
    if ($value === null) {
        return 'null';
    }
    
    if (is_bool($value)) {
        return 'boolean';
    }
    
    if (is_int($value)) {
        return 'integer';
    }
    
    if (is_float($value)) {
        return 'float';
    }
    
    if (is_array($value) || is_object($value)) {
        return 'json';
    }
    
    // Check if it's a date
    if (preg_match('/^\d{4}-\d{2}-\d{2}/', $value)) {
        return 'date';
    }
    
    // Check if it's a timestamp
    if (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}/', $value)) {
        return 'timestamp';
    }
    
    return 'string';
}

/**
 * Convert PHP array to HTML table
 */
function arrayToTable($data, $classes = 'table table-striped') {
    if (empty($data)) {
        return '<p>No data available</p>';
    }
    
    $html = '<table class="' . h($classes) . '">';
    
    // Header
    $html .= '<thead><tr>';
    foreach (array_keys($data[0]) as $header) {
        $html .= '<th>' . h($header) . '</th>';
    }
    $html .= '</tr></thead>';
    
    // Body
    $html .= '<tbody>';
    foreach ($data as $row) {
        $html .= '<tr>';
        foreach ($row as $cell) {
            $html .= '<td>' . h($cell) . '</td>';
        }
        $html .= '</tr>';
    }
    $html .= '</tbody>';
    
    $html .= '</table>';
    
    return $html;
}

/**
 * Generate unique ID
 */
function generateId($prefix = '') {
    return $prefix . uniqid() . mt_rand(1000, 9999);
}

/**
 * Check if value is empty but allow 0
 */
function isReallyEmpty($value) {
    return $value === null || $value === '' || $value === [];
}

/**
 * Safe array get with default value
 */
function arrayGet($array, $key, $default = null) {
    return isset($array[$key]) ? $array[$key] : $default;
}

/**
 * Convert snake_case to Title Case
 */
function snakeToTitle($string) {
    return ucwords(str_replace('_', ' ', $string));
}
?>
